#!/usr/bin/env node
import { Server } from '@modelcontextprotocol/sdk/server/index.js';
import { StdioServerTransport } from '@modelcontextprotocol/sdk/server/stdio.js';
import {
    CallToolRequestSchema,
    ErrorCode,
    ListToolsRequestSchema,
    McpError,
} from '@modelcontextprotocol/sdk/types.js';
import { z } from 'zod';
import { RuleIndexer } from './rules/indexer.js';
import { Composer } from './rules/composer.js';
import * as path from 'path';

// --- Configuration ---
const ROOT_DIR = process.cwd(); // Assume running from project root

// --- Initialization ---
const indexer = new RuleIndexer(ROOT_DIR);
const composer = new Composer(ROOT_DIR);

// Init index on startup (async but we assume fast enough or lazy)
indexer.init().catch(err => console.error('Failed to init index:', err));

const server = new Server(
    {
        name: 'awing-rules-claudecode-mcp',
        version: '0.1.0',
    },
    {
        capabilities: {
            tools: {},
        },
    }
);

// --- Tool Definitions ---

server.setRequestHandler(ListToolsRequestSchema, async () => {
    return {
        tools: [
            {
                name: 'rules_search',
                description: 'Search for relevant project rules based on query and context.',
                inputSchema: {
                    type: 'object',
                    properties: {
                        query: { type: 'string', description: 'The search query or user task description' },
                        openFiles: { type: 'array', items: { type: 'string' }, description: 'List of currently open file paths' },
                        changedFiles: { type: 'array', items: { type: 'string' }, description: 'List of recently changed file paths' },
                        tags: { type: 'array', items: { type: 'string' }, description: 'Filter by tags' },
                        limit: { type: 'number', description: 'Max number of results to return (default 6)' },
                        minScore: { type: 'number', description: 'Minimum score threshold (default 0.15)' }
                    },
                    required: ['query'],
                },
            },
            {
                name: 'rules_get',
                description: 'Get details of a specific rule by ID or path.',
                inputSchema: {
                    type: 'object',
                    properties: {
                        id: { type: 'string' },
                        path: { type: 'string' },
                        mode: { type: 'string', enum: ['full', 'snippet', 'sections'], default: 'snippet' }
                    },
                },
            },
            {
                name: 'rules_compose',
                description: 'Compose a rule bundle from selected rules and the base rule.',
                inputSchema: {
                    type: 'object',
                    properties: {
                        selected: {
                            type: 'array',
                            items: {
                                type: 'object',
                                properties: {
                                    id: { type: 'string' },
                                    path: { type: 'string' }
                                }
                            }
                        },
                        mode: { type: 'string', enum: ['full', 'snippet'], default: 'snippet' },
                        dedupe: { type: 'boolean', default: true },
                        maxChars: { type: 'number', default: 12000 }
                    },
                    required: ['selected'],
                },
            },
            {
                name: 'rules_refresh',
                description: 'Refresh the rule index from disk.',
                inputSchema: {
                    type: 'object',
                    properties: {},
                },
            },
        ],
    };
});

// --- Tool Handlers ---

server.setRequestHandler(CallToolRequestSchema, async (request) => {
    const { name, arguments: args } = request.params;

    if (name === 'rules_refresh') {
        await indexer.refresh();
        return {
            content: [{ type: 'text', text: 'Rule index refreshed successfully.' }]
        };
    }

    if (name === 'rules_search') {
        const input = z.object({
            query: z.string(),
            openFiles: z.array(z.string()).optional(),
            changedFiles: z.array(z.string()).optional(),
            tags: z.array(z.string()).optional(),
            limit: z.number().optional(),
            minScore: z.number().optional()
        }).parse(args);

        const results = indexer.search(input.query, {
            openFiles: input.openFiles,
            changedFiles: input.changedFiles,
            tags: input.tags,
            limit: input.limit,
            minScore: input.minScore
        });

        return {
            content: [
                {
                    type: 'text',
                    text: JSON.stringify(results.map(r => ({
                        id: r.rule.id,
                        path: r.rule.relativePath,
                        title: r.rule.title,
                        score: r.score.toFixed(2),
                        tags: r.rule.tags,
                        why: `Text:${r.scoreBreakdown.text.toFixed(2)} Path:${r.scoreBreakdown.path.toFixed(2)} Tag:${r.scoreBreakdown.tag.toFixed(2)}`
                    })), null, 2),
                },
            ],
        };
    }

    if (name === 'rules_get') {
        const input = z.object({
            id: z.string().optional(),
            path: z.string().optional(),
            mode: z.string().optional(),
        }).parse(args);

        const rule = indexer.getRuleByIdOrPath(input.id || input.path || '');
        if (!rule) {
            return { isError: true, content: [{ type: 'text', text: 'Rule not found' }] };
        }

        return {
            content: [
                {
                    type: 'text',
                    text: JSON.stringify({
                        id: rule.id,
                        path: rule.relativePath,
                        title: rule.title,
                        content: rule.content // TODO: Apply mode/snippet logic if needed
                    }, null, 2),
                },
            ],
        };
    }

    if (name === 'rules_compose') {
        const input = z.object({
            selected: z.array(z.object({ id: z.string().optional(), path: z.string().optional() })),
            mode: z.enum(['full', 'snippet']).optional(),
            dedupe: z.boolean().optional(),
            maxChars: z.number().optional()
        }).parse(args);

        // Resolve rules
        const resolvedRules = input.selected.map(sel => {
            const rule = indexer.getRuleByIdOrPath(sel.id || sel.path || '');
            return { ...sel, rule };
        });

        const bundle = await composer.compose(resolvedRules, {
            mode: input.mode,
            dedupe: input.dedupe,
            maxChars: input.maxChars
        });

        return {
            content: [
                {
                    type: 'text',
                    text: bundle.content,
                },
            ],
        };
    }

    throw new McpError(ErrorCode.MethodNotFound, `Unknown tool: ${name}`);
});

async function runServer() {
    const transport = new StdioServerTransport();
    await server.connect(transport);
    console.error('Moving Rules MCP Server running on stdio');
}

runServer().catch((error) => {
    console.error('Fatal error running server:', error);
    process.exit(1);
});
