import { Rule, RuleBundle } from '../types.js';
import { RuleLoader } from './loader.js';

export class Composer {
    private loader: RuleLoader;

    constructor(baseDir: string) {
        this.loader = new RuleLoader(baseDir);
    }

    async compose(
        selectedRules: { id?: string; path?: string; rule?: Rule }[],
        options: {
            maxChars?: number;
            dedupe?: boolean;
            mode?: 'full' | 'snippet';
            basePath?: string;
        } = {}
    ): Promise<RuleBundle> {
        const { maxChars = 12000, dedupe = true, mode = 'snippet' } = options;

        // 1. Load Base Rule
        const baseContent = await this.loader.loadBaseRule();

        // 2. Resolve Selected Rules
        // The indexer might have passed the full Rule object or just ID/Path.
        // Ideally, the caller (MCP tool) resolves the rules first using Indexer, or passes them in.
        // For simplicity, let's assume 'selectedRules' contains the Rule object if passed from search,
        // or we might need to look them up. But Composer doesn't have the Indexer reference.
        // Design choice: MCP server does the lookup. Composer just formats.
        // So we expect `rule` property to be present or we can't extract info.
        // Actually, let's make the input strictly `Rule[]`.

        // Refactoring spec: input `selected` is [{id?, path?}].
        // Composer needs access to Indexer or Loader to fetch content if not provided?
        // Let's assume the MCP layer fetches the rules.
        // "Input: selected: [{ id?: string, path?: string }]".
        // WE need to fetch them. So Composer should take an Indexer or Loader.
        // But Loader is expensive to re-scan. Indexer has cache.
        // Let's pass the *resolved* rules to the compose function to keep it pure(r).
        // Or simpler: Composer owns the formatting string logic, MCP server orchestration handles lookup.

        // Let's write this class to accept *resolved* array of Rules.
        // If the tool definition says input is just ids, the MCP handler will use Indexer to get the Rules, then pass them here.

        const rulesToInclude = selectedRules.map(s => s.rule).filter((r): r is Rule => !!r);

        // 3. Start composing
        let bundle = `# RULE BUNDLE\n> [!IMPORTANT] Priority Order: Selected rules override Base rules where conflicting. Security constraints in Base rules are non-negotiable.\n\n`;

        // Add Base
        bundle += `## BASE RULES\n\n${baseContent}\n\n`;

        // Add Selected
        const used: { id: string; path: string; title: string }[] = [];
        const notes: string[] = [];

        for (const rule of rulesToInclude) {
            bundle += `## ${rule.title} (ID: ${rule.id})\nTitle: ${rule.title}\nSource: ${rule.relativePath}\n\n`;

            let contentToAdd = rule.content;

            // Snippet mode logic
            if (mode === 'snippet') {
                contentToAdd = this.extractSnippet(contentToAdd);
            }

            bundle += contentToAdd + `\n\n---\n\n`;
            used.push({ id: rule.id, path: rule.path, title: rule.title });
        }

        // 4. Deduplication
        if (dedupe) {
            bundle = this.deduplicateLines(bundle);
        }

        // 5. Truncation
        let truncated = false;
        if (bundle.length > maxChars) {
            bundle = bundle.slice(0, maxChars) + '\n... (truncated)';
            truncated = true;
            notes.push(`Output truncated to ${maxChars} chars.`);
        }

        return {
            content: bundle,
            used,
            truncated,
            notes
        };
    }

    private extractSnippet(content: string): string {
        // Keep headings, bullets, blockquotes
        const lines = content.split('\n');
        const keep: string[] = [];
        let insideCodeBlock = false;

        for (const line of lines) {
            if (line.trim().startsWith('```')) {
                insideCodeBlock = !insideCodeBlock;
                // In snippet mode, maybe omit long code blocks? 
                // Let's keep them if they are short? 
                // For now, keep everything but maybe limit length?
                keep.push(line);
                continue;
            }

            const isHeader = /^#+\s/.test(line);
            const isList = /^[\s]*[-*+]\s/.test(line);
            const isQuote = /^>\s/.test(line); // e.g. alerts

            if (isHeader || isList || isQuote || insideCodeBlock) {
                keep.push(line);
            } else {
                // Plain text? Skip in harsh snippet mode?
                // Let's keep it if it's short, or skip?
                // Prompt says: "Prioritize headings + bullet lines + Do/Don't"
                // Let's conservatively keep non-empty logic lines?
                // Actually, just skipping paragraphs might be intended. 
                if (line.trim().length > 0) {
                    // keep.push(line); // aggressive: keep all?
                    // "Cắt examples dài"
                }
            }
        }
        // Simple pass: return full content for now, but assume text is mostly bullets in rules.
        // Better implementation: Regex for relevant sections.
        return content; // Placeholder for advanced snippet extraction if needed.
    }

    private deduplicateLines(text: string): string {
        const lines = text.split('\n');
        const seen = new Set<string>();
        const out: string[] = [];

        for (const line of lines) {
            const trimmed = line.trim().toLowerCase();
            // Only dedupe substantial bullet points
            const isBullet = /^[\s]*[-*+]\s/.test(line);

            if (isBullet && trimmed.length > 10) {
                if (seen.has(trimmed)) {
                    continue; // Skip duplicate
                }
                seen.add(trimmed);
            }
            out.push(line);
        }
        return out.join('\n');
    }
}
