import { RuleLoader } from './loader.js';
import { Scorer } from './scorer.js';
export class RuleIndexer {
    loader;
    scorer;
    cachedRules = [];
    lastIndexed = 0;
    constructor(baseDir) {
        this.loader = new RuleLoader(baseDir);
        this.scorer = new Scorer();
    }
    async init() {
        await this.refresh();
    }
    async refresh() {
        this.cachedRules = await this.loader.loadAllRules();
        this.scorer.indexRules(this.cachedRules);
        this.lastIndexed = Date.now();
    }
    async getBaseRule() {
        return this.loader.loadBaseRule();
    }
    getRuleByIdOrPath(idOrPath) {
        // Try ID match
        let rule = this.cachedRules.find(r => r.id === idOrPath);
        if (rule)
            return rule;
        // Try Path match (exact or relative)
        // normalizing slashes
        const normalized = idOrPath.replace(/\\/g, '/');
        return this.cachedRules.find(r => r.path.replace(/\\/g, '/').endsWith(normalized));
    }
    search(query, options = {}) {
        const { openFiles = [], changedFiles = [], tags = [], limit = 6, minScore = 0.15 } = options;
        const results = this.cachedRules.map(rule => this.scorer.scoreRule(rule, query, openFiles, changedFiles, tags));
        // Sort by score desc
        const sorted = results
            .filter(r => r.score >= minScore)
            .sort((a, b) => b.score - a.score);
        // Diversification and Deduplication logic
        const diversified = [];
        const categoryCounts = {};
        const seenIds = new Set();
        for (const res of sorted) {
            // Deduplicate by ID
            if (seenIds.has(res.rule.id))
                continue;
            // Determine category from relative path (first dir)
            const parts = res.rule.relativePath.split(/[/\\]/);
            const category = parts.length > 1 ? parts[0] : 'root';
            if ((categoryCounts[category] || 0) < 2) {
                diversified.push(res);
                categoryCounts[category] = (categoryCounts[category] || 0) + 1;
                seenIds.add(res.rule.id);
            }
            if (diversified.length >= limit)
                break;
        }
        return diversified;
    }
}
