import { Scorer } from '../rules/scorer.js';
import { Composer } from '../rules/composer.js';
import { RuleIndexer } from '../rules/indexer.js';
import { Rule } from '../types.js';
import * as assert from 'assert';

// Simple test helper
function test(name: string, fn: () => void) {
    try {
        fn();
        console.log(`✅ ${name}`);
    } catch (e) {
        console.error(`❌ ${name}`);
        console.error(e);
    }
}

const mockRule: Rule = {
    id: 'test-rule',
    path: '/abs/test/frontend/rule.md',
    relativePath: 'frontend/rule.md',
    title: 'Test Rule',
    content: 'Rule content',
    tags: ['react', 'testing'],
    priority: 80,
    paths: ['**/*.tsx'],
    applies_when: [],
    avoid: [],
    lastModified: 0
};

async function runTests() {
    console.log('--- Running Scorer Tests ---');

    // Scorer Path Tests
    const scorer = new Scorer();
    // We need to index rules first to initialize vectors if we test text score, 
    // but path score is independent.
    scorer.indexRules([mockRule]);

    test('Path Score: Exact Match', () => {
        const score = (scorer as any).calculatePathScore(mockRule, ['/src/app.tsx'], ['/src/component.tsx']);
        // Changed file matches glob **/*.tsx
        assert.ok(score >= 1.0, 'Should be 1.0 for changed file match');
    });

    test('Path Score: Open Match', () => {
        const score = (scorer as any).calculatePathScore(mockRule, ['/src/app.tsx'], []);
        assert.ok(score >= 0.6, 'Should be 0.6 for open file match');
    });

    test('Path Score: No Match', () => {
        const score = (scorer as any).calculatePathScore(mockRule, ['/src/main.py'], []);
        assert.strictEqual(score, 0, 'Should be 0 for no match');
    });

    console.log('--- Running Composer Tests ---');
    const composer = new Composer('.');

    test('Deduplication', () => {
        const text = `
- Keep this
- Dedupe this
- Dedupe this
- Keep this too
`;
        const result = (composer as any).deduplicateLines(text);
        const lines = result.split('\n').map((l: string) => l.trim()).filter((l: string) => l);
        assert.strictEqual(lines.length, 3, 'Should remove 1 duplicate line');
    });

    console.log('--- Real File Indexing Test ---');
    const indexer = new RuleIndexer(process.cwd());
    await indexer.init(); // Load real files

    const results = indexer.search('react component', {
        openFiles: ['src/App.tsx']
    });

    console.log(`Found ${results.length} rules for "react component"`);
    results.forEach(r => console.log(`- ${r.rule.title} (Score: ${r.score.toFixed(2)})`));

    // Check if we found the new rules
    const foundReact = results.some(r => r.rule.id === 'frontend-react' || r.rule.title.includes('React'));
    assert.ok(foundReact, 'Should find React rules in sample files');

    const baseContent = await indexer.getBaseRule();
    assert.ok(baseContent.length > 0, 'Base rule should be loaded');

    console.log('--- All Tests Finished ---');
}

runTests().catch(console.error);
