import * as fs from 'fs/promises';
import * as path from 'path';
import matter from 'gray-matter';
import { Rule, RuleFrontmatter } from '../types.js';
import { findMarkdownFiles } from '../utils/glob.js';

export class RuleLoader {
    private rootDir: string;

    constructor(rootDir: string) {
        this.rootDir = path.resolve(rootDir);
    }

    async loadAllRules(): Promise<Rule[]> {
        const files = await findMarkdownFiles(this.rootDir);
        const rules: Rule[] = [];

        for (const file of files) {
            // Note: We used to skip base.md here, but we want it indexed for search.
            // if (path.basename(file).toLowerCase() === 'base.md') continue;

            try {
                const rule = await this.parseRule(file);
                if (rule) rules.push(rule);
            } catch (error) {
                console.error(`Failed to parse rule file: ${file}`, error);
            }
        }

        return rules;
    }

    async loadBaseRule(): Promise<string> {
        const basePath = path.join(this.rootDir, 'base.md');
        try {
            const content = await fs.readFile(basePath, 'utf-8');
            return content;
        } catch (error) {
            // If base.md doesn't exist, return empty string or default warning
            return '';
        }
    }

    private async parseRule(filePath: string): Promise<Rule | null> {
        const rawContent = await fs.readFile(filePath, 'utf-8');
        const { data, content } = matter(rawContent);
        const fm = data as RuleFrontmatter;
        const stats = await fs.stat(filePath);

        // Filter out rules explicitly marked to be avoided if necessary? 
        // No, 'avoid' is a string[] field for query matching, not a flag to ignore the file itself basically. 
        // Unless frontmatter is drastically invalid.

        const relativePath = path.relative(this.rootDir, filePath);
        const id = fm.id || relativePath.replace(/\\/g, '/').replace(/\.md$/, '');
        const title = fm.title || path.basename(filePath, '.md');

        return {
            id,
            path: filePath,
            relativePath,
            title,
            content,
            tags: fm.tags || [],
            priority: typeof fm.priority === 'number' ? fm.priority : 50,
            paths: fm.paths || [],
            applies_when: fm.applies_when || [],
            avoid: fm.avoid || [],
            lastModified: stats.mtimeMs
        };
    }
}
