#!/usr/bin/env node
import { Server } from '@modelcontextprotocol/sdk/server/index.js';
import { StdioServerTransport } from '@modelcontextprotocol/sdk/server/stdio.js';
import { CallToolRequestSchema, ListToolsRequestSchema } from '@modelcontextprotocol/sdk/types.js';
import { GraphQLClient } from 'graphql-request';
import { tools, executeQuery } from './tools/index.js';
// Configuration from environment variables
const GRAPHQL_ENDPOINT = process.env.GRAPHQL_ENDPOINT || 'https://acm.awifi.com.vn/graphql';
const AUTH_TOKEN = process.env.AUTH_TOKEN || '';
const WORKSPACE_ID = process.env.WORKSPACE_ID || '11';
// Create GraphQL client
const graphqlClient = new GraphQLClient(GRAPHQL_ENDPOINT, {
    headers: {
        Authorization: `Bearer ${AUTH_TOKEN}`,
        workspaceid: WORKSPACE_ID
    }
});
// Create MCP server
const server = new Server({
    name: 'acm-graphql-mcp',
    version: '1.0.0'
}, {
    capabilities: {
        tools: {}
    }
});
// List available tools
server.setRequestHandler(ListToolsRequestSchema, async () => {
    return {
        tools: tools
    };
});
// Handle tool calls
server.setRequestHandler(CallToolRequestSchema, async (request) => {
    const { name, arguments: args } = request.params;
    try {
        const result = await executeQuery(graphqlClient, name, args);
        return {
            content: [
                {
                    type: 'text',
                    text: JSON.stringify(result, null, 2)
                }
            ]
        };
    }
    catch (error) {
        const errorMessage = error instanceof Error ? error.message : String(error);
        return {
            content: [
                {
                    type: 'text',
                    text: `Error: ${errorMessage}`
                }
            ],
            isError: true
        };
    }
});
// Start the server
async function main() {
    const transport = new StdioServerTransport();
    await server.connect(transport);
    console.error('ACM GraphQL MCP Server running on stdio');
}
main().catch((error) => {
    console.error('Server error:', error);
    process.exit(1);
});
