
import { Client } from "@modelcontextprotocol/sdk/client/index.js";
import { StdioClientTransport } from "@modelcontextprotocol/sdk/client/stdio.js";
import dotenv from "dotenv";
import path from "path";
import { fileURLToPath } from "url";

// Load environment variables
function loadEnv() {
    const __dirname = path.dirname(fileURLToPath(import.meta.url));
    const envPath = path.resolve(__dirname, "../../.env");
    const result = dotenv.config({ path: envPath });
    if (result.error) {
        console.warn(`Warning: .env file not found at ${envPath}`);
    }
    return process.env;
}

async function main() {
    console.log("=== Test Jira Boards Resource ===");

    const envVars = loadEnv();
    const serverPath = "/home/lethdat/projects/awing/mcp-jira-server/dist/index.js";

    const client = new Client({ name: "test-client-boards", version: "1.0.0" }, {
        capabilities: { promts: {}, resources: {}, tools: {} }
    });

    const transport = new StdioClientTransport({
        command: "node",
        args: [serverPath],
        env: { ...process.env, ...envVars }
    });

    await client.connect(transport);
    console.log("Connected to MCP server");

    try {
        // 1. Test List Boards
        console.log(`\nTesting jira://boards...`);
        const boardsRes = await client.readResource({ uri: `jira://boards` });
        const boardsData = JSON.parse(String(boardsRes.contents[0].text));
        console.log(`Found ${boardsData.metadata?.total || boardsData.boards?.length || 0} boards.`);

        if (boardsData.boards && boardsData.boards.length > 0) {
            const firstBoard = boardsData.boards[1];
            const boardId = firstBoard.id;
            console.log(`Testing with Board: ${firstBoard.name} (ID: ${boardId})`);

            // 2. Test Board Details
            console.log(`\nTesting jira://boards/${boardId}...`);
            const boardDetailRes = await client.readResource({ uri: `jira://boards/${boardId}` });
            const boardDetailData = JSON.parse(String(boardDetailRes.contents[0].text));
            console.log("Board Detail:", JSON.stringify(boardDetailData.board[0]?.name));

            // 3. Test Board Issues
            console.log(`\nTesting jira://boards/${boardId}/issues...`);
            const boardIssuesRes = await client.readResource({ uri: `jira://boards/${boardId}/issues` });
            const boardIssuesData = JSON.parse(String(boardIssuesRes.contents[0].text));
            console.log(`Found ${boardIssuesData.metadata?.total || boardIssuesData.issues?.length || 0} issues on board.`);

            // 4. Test Board Sprints (only if it's a scrum board, but safe to call anyway usually)
            if (firstBoard.type === 'scrum') {
                console.log(`\nTesting jira://boards/${boardId}/sprints...`);
                const boardSprintsRes = await client.readResource({ uri: `jira://boards/${boardId}/sprints` });
                const boardSprintsData = JSON.parse(String(boardSprintsRes.contents[0].text));
                console.log(`Found ${boardSprintsData.metadata?.total || boardSprintsData.sprints?.length || 0} sprints on board.`);
            } else {
                console.log(`\nSkipping sprints test (Board type: ${firstBoard.type})`);
            }
        }
    } catch (error) {
        console.error(`Error testing boards:`, error);
    }

    client.close();
}

main().catch(console.error);
