import { Tool } from '@modelcontextprotocol/sdk/types.js'
import { analyzeDesignSystem } from '../core/design-system.core.js'
import { analyzeStorybookUsage } from '../core/storybook.core.js'
import { parseUIIntent } from '../core/intent.core.js'
import { inferProjectContext } from '../context/project.context.js'
import { generateReactCode } from '../core/codegen.core.js'
import { loadUILibraryContext } from '../context/ui-library.context.js'

// Tool definitions theo chuẩn MCP
export const tools: Tool[] = [
    {
        name: 'analyze_design_system',
        description: 'Phân tích các components trong design system của dự án',
        inputSchema: {
            type: 'object',
            properties: {
                libraryName: {
                    type: 'string',
                    description: 'Tên của UI library cần phân tích (mặc định: awing-library)'
                }
            }
        }
    },
    {
        name: 'analyze_storybook',
        description: 'Phân tích usage của một component trong Storybook',
        inputSchema: {
            type: 'object',
            properties: {
                component: {
                    type: 'string',
                    description: 'Tên component cần phân tích'
                }
            },
            required: ['component']
        }
    },
    {
        name: 'parse_ui_intent',
        description: 'Phân tích ý định UI từ prompt của người dùng để xác định components cần dùng',
        inputSchema: {
            type: 'object',
            properties: {
                prompt: {
                    type: 'string',
                    description: 'Mô tả UI mà người dùng muốn tạo'
                }
            },
            required: ['prompt']
        }
    },
    {
        name: 'get_project_context',
        description: 'Lấy thông tin context của dự án hiện tại (framework, dependencies, structure)',
        inputSchema: {
            type: 'object',
            properties: {}
        }
    },
    {
        name: 'generate_ui_code',
        description: 'Generate React code dựa trên UI intent đã được parse',
        inputSchema: {
            type: 'object',
            properties: {
                description: {
                    type: 'string',
                    description: 'Mô tả UI'
                },
                components: {
                    type: 'array',
                    items: { type: 'string' },
                    description: 'Danh sách components cần sử dụng'
                }
            },
            required: ['description', 'components']
        }
    }
]

// Execute tool function
export async function executeTool(
    toolName: string,
    args: Record<string, unknown>
): Promise<unknown> {
    switch (toolName) {
        case 'analyze_design_system': {
            const libraryName = (args.libraryName as string) || 'awing-library'
            const ctx = loadUILibraryContext(libraryName)
            return analyzeDesignSystem(ctx)
        }

        case 'analyze_storybook': {
            const component = args.component as string
            return analyzeStorybookUsage(component)
        }

        case 'parse_ui_intent': {
            const prompt = args.prompt as string
            return parseUIIntent(prompt)
        }

        case 'get_project_context': {
            return inferProjectContext()
        }

        case 'generate_ui_code': {
            const intent = {
                description: args.description as string,
                components: args.components as string[]
            }
            return generateReactCode(intent)
        }

        default:
            throw new Error(`Unknown tool: ${toolName}`)
    }
}
